﻿
/****************************************************************************/
/*Copyright (c) 2011, Florent DEVILLE.                                      */
/*All rights reserved.                                                      */
/*                                                                          */
/*Redistribution and use in source and binary forms, with or without        */
/*modification, are permitted provided that the following conditions        */
/*are met:                                                                  */
/*                                                                          */
/* - Redistributions of source code must retain the above copyright         */
/*notice, this list of conditions and the following disclaimer.             */
/* - Redistributions in binary form must reproduce the above                */
/*copyright notice, this list of conditions and the following               */
/*disclaimer in the documentation and/or other materials provided           */
/*with the distribution.                                                    */
/* - The names of its contributors cannot be used to endorse or promote     */
/*products derived from this software without specific prior written        */
/*permission.                                                               */
/* - The source code cannot be used for commercial purposes without         */
/*its contributors' permission.                                             */
/*                                                                          */
/*THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       */
/*"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT         */
/*LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS         */
/*FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE            */
/*COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,       */
/*INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,      */
/*BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;          */
/*LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER          */
/*CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT        */
/*LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN         */
/*ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE           */
/*POSSIBILITY OF SUCH DAMAGE.                                               */
/****************************************************************************/

using GE.Visualisation;
using GE.Physics.Shapes;
using GE.Physics;
using GE.Manager;
using Microsoft.Xna.Framework;

namespace GE.World.Entities
{
    class WallShooter : EnemyEntity
    {
        enum eState
        {
            eStateOpen,
            eStateClose,
            eStateOpening,
            eStateClosing,
            eStateNone,
            eStateCount
        }

        eState _innerState;

        int _iIdTextureClose;
        int _iIdSpriteClose;
        int _iIdTextureOpen;
        int _iIdSpriteOpen;
        int _iIdAnimOpening;
        int _iIdAnimClosing;
        int _iIdTextureBullet;
        int _iIdSpriteBullet;

        int _iTimeClosed;
        int _iStartTimeClose;

        int _iTimeLastShoot;
        const int TIME_BETWEEN_SHOOT = 500;

        int _iDirection;

        //StaticShapeRectangle _shape;
        bool _bAnimationOver;
        int _iAnimCurrentFrame;
        int _iAnimCurrentTime;

        bool _bPreState;
        int _iShootCount;

        Vector2 _v2OffsetPositionOpen;
        Vector2 _v2BulletSpawnPosition;

        /// <summary>
        /// Id of the explosion animation
        /// </summary>
        int _iIdAnimationExplosion;

        /// <summary>
        /// Explosion animation offset
        /// </summary>
        Vector2 _v2PositionExplosionOffset;

        /// <summary>
        /// Bullet damages
        /// </summary>
        int _iBulletDamages;

        #region Properties
#if !GAME
        public static string EDITOR_TILESET { get { return "enemysheet.xml"; } }
        public static string EDITOR_SPRITE { get { return "wall_shooter_4"; } }
#endif
        public int TimeClosed { set { _iTimeClosed = value; } }
        public int Direction { set { _iDirection = value; } }
        public int BulletDamages { set { _iBulletDamages = value; } }
        #endregion

        /// <summary>
        /// Constructor
        /// </summary>
        public WallShooter()
            :base()
        {
            _innerState = eState.eStateNone;
            _iIdAnimClosing = -1;
            _iIdAnimOpening = -1;
            _iIdSpriteClose = -1;
            _iIdSpriteOpen = -1;
            _iIdTextureClose = -1;
            _iIdTextureOpen = -1;
            _iTimeClosed = -1;
            _iStartTimeClose = -1;
            _iTimeLastShoot = -1;
            _iDirection = 1;
            _iAnimCurrentTime = -1;
            _iAnimCurrentFrame = -1;
            _iIdSpriteBullet = -1;
            _iIdTextureBullet = -1;
            _iIdAnimationExplosion = -1;
            _v2PositionExplosionOffset = Vector2.Zero;
            _entityType = eEntityType.eEntityEnemy;
        }

        /// <summary>
        /// Initialise the entity. Load all the data which are not loaded during the level loading
        /// </summary>
        public override void init()
        {
            _iIdTextureClose = Visu.Instance.loadTilset("enemysheet.xml");
            _iIdSpriteClose = Visu.Instance.getSpriteId(_iIdTextureClose, "wall_shooter_1");
            _iIdTextureOpen = _iIdTextureClose;
            _iIdSpriteOpen = Visu.Instance.getSpriteId(_iIdTextureClose, "wall_shooter_4");
            _iIdAnimClosing = Visu.Instance.getAnimationID("Wall_Shooter_Close");
            _iIdAnimOpening = Visu.Instance.getAnimationID("Wall_Shooter_Open");
            _iIdAnimationExplosion = Visu.Instance.getAnimationID("Little_Explosion");
            _iIdTextureBullet = _iIdTextureClose;
            _iIdSpriteBullet = Visu.Instance.getSpriteId(_iIdTextureBullet, "wall_shooter_bullet");

            //create the collision shape
            int shapeWidth = Visu.Instance.getSpriteWidth(_iIdTextureClose, _iIdSpriteClose);
            int shapeHeight = Visu.Instance.getSpriteHeight(_iIdTextureClose, _iIdSpriteClose);
            _shape = Physics.Physics.Instance.createStaticRectangle(shapeWidth, shapeHeight, Vector2.Zero, _v2Position, 0, this);
            _shape._iGroup = (int)ePhysicGroup.ePhysicEnemy;

            //calculate the offsets
            int iWidthOpen = Visu.Instance.getSpriteWidth(_iIdTextureOpen, _iIdSpriteOpen);
            int iBulletHeight = Visu.Instance.getSpriteHeight(_iIdTextureBullet, _iIdSpriteBullet);

            _v2OffsetPositionOpen = new Vector2(shapeWidth - iWidthOpen, 0);
            _v2BulletSpawnPosition = new Vector2(iWidthOpen, shapeHeight / 2 - iBulletHeight / 2);
            if (_iDirection > 0)
                _v2BulletSpawnPosition = _v2Position + _v2BulletSpawnPosition;
            else
                _v2BulletSpawnPosition = _v2Position + new Vector2(-iWidthOpen+8, shapeHeight / 2 - iBulletHeight / 2);

            int height = Visu.Instance.getSpriteHeight(_iIdTextureOpen, _iIdSpriteOpen);
            _v2PositionExplosionOffset = new Vector2(0, height / 2);

            base.init();
        }

        /// <summary>
        /// Activator
        /// </summary>
        public override void activate()
        {
            _innerState = eState.eStateClose;
            _bActive = true;
            _bAnimationOver = false;
            _bPreState = true;

            _shape._bCollisionEnable = true;
        }

        public override void update()
        {
            switch (_innerState)
            {
                case eState.eStateClose:
                    updateStateClose();
                    break;
                case eState.eStateClosing:
                    updateStateClosing();
                    break;
                case eState.eStateOpen:
                    updateStateOpen();
                    break;
                case eState.eStateOpening:
                    updateStateOpening();
                    break;
            }

            CollisionResult res = Physics.Physics.Instance.checkFirstRegisteredCollisionEx(_shape, (int)ePhysicGroup.ePhysicPlayer);
            if (res != null) res.Entity.hurt(_iDamages);
            
        }

        private void updateStateClose()
        {
            if (_bPreState)
            {
                _bPreState = false;
                _iStartTimeClose = TimeClock.Clock.instance.millisecs;
            }

            if (TimeClock.Clock.instance.millisecs > _iStartTimeClose + _iTimeClosed)
            {
                _innerState = eState.eStateOpening;
                _bPreState = true;
            }
        }

        private void updateStateOpening()
        {
            if (_bAnimationOver)
            {
                _innerState = eState.eStateOpen;
                _bPreState = true;
            }
        }

        private void updateStateOpen()
        {
            if (_bPreState)
            {
                _bPreState = false;
                _iTimeLastShoot = TimeClock.Clock.instance.millisecs;
                _iShootCount = 0;
            }

            if (_iShootCount == 4 && TimeClock.Clock.instance.millisecs > _iTimeLastShoot + TIME_BETWEEN_SHOOT)
            {
                _innerState = eState.eStateClosing;
                _bPreState = false;
            }

            if (TimeClock.Clock.instance.millisecs > _iTimeLastShoot + TIME_BETWEEN_SHOOT)
            {
                const int BULLET_LIFE_TIME = 2000;
                switch (_iShootCount)
                {
                    case 0:
                        BulletManager.Instance.activate(_iIdTextureBullet, _iIdSpriteBullet, BULLET_LIFE_TIME, _iBulletDamages, new Vector2(_iDirection * 2, -3), _v2BulletSpawnPosition, eSide.eSideEnemy);
                        break;
                    case 1:
                        BulletManager.Instance.activate(_iIdTextureBullet, _iIdSpriteBullet, BULLET_LIFE_TIME, _iBulletDamages, new Vector2(_iDirection * 3, -1), _v2BulletSpawnPosition, eSide.eSideEnemy);
                        break;
                    case 2:
                        BulletManager.Instance.activate(_iIdTextureBullet, _iIdSpriteBullet, BULLET_LIFE_TIME, _iBulletDamages, new Vector2(_iDirection * 3, 1), _v2BulletSpawnPosition, eSide.eSideEnemy);
                        break;
                    case 3:
                        BulletManager.Instance.activate(_iIdTextureBullet, _iIdSpriteBullet, BULLET_LIFE_TIME, _iBulletDamages, new Vector2(_iDirection * 2, 3), _v2BulletSpawnPosition, eSide.eSideEnemy);
                        break;
                }
                _iShootCount++;
                _iTimeLastShoot = TimeClock.Clock.instance.millisecs;
            }

        }

        private void updateStateClosing()
        {
            if (_bAnimationOver)
            {
                _innerState = eState.eStateClose;
                _bPreState = true;
            }
        }

        public override void render()
        {

            switch (_innerState)
            {
                case eState.eStateClose:
                    if (_iDirection > 0)
                        Visu.Instance.displaySprite(_iIdTextureClose, _iIdSpriteClose, ScreenPosition, Microsoft.Xna.Framework.Graphics.SpriteEffects.FlipHorizontally);
                    else
                        Visu.Instance.displaySprite(_iIdTextureClose, _iIdSpriteClose, ScreenPosition);
                    break;
                case eState.eStateClosing:
                    if (_iDirection > 0)
                        _bAnimationOver = Visu.Instance.displayAnimation(_iIdAnimClosing, ScreenPosition + _v2OffsetPositionOpen - new Vector2(4, 0), ref _iAnimCurrentFrame, ref _iAnimCurrentTime, Microsoft.Xna.Framework.Graphics.SpriteEffects.FlipHorizontally);
                    else
                        _bAnimationOver = Visu.Instance.displayAnimation(_iIdAnimClosing, ScreenPosition, ref _iAnimCurrentFrame, ref _iAnimCurrentTime);
                    break;
                case eState.eStateOpen:
                    if(_iDirection > 0)
                        Visu.Instance.displaySprite(_iIdTextureOpen, _iIdSpriteOpen, ScreenPosition, Microsoft.Xna.Framework.Graphics.SpriteEffects.FlipHorizontally);
                    else
                        Visu.Instance.displaySprite(_iIdTextureOpen, _iIdSpriteOpen, ScreenPosition+_v2OffsetPositionOpen);
                    break;
                case eState.eStateOpening:
                    if (_iDirection > 0)
                        _bAnimationOver = Visu.Instance.displayAnimation(_iIdAnimOpening, ScreenPosition + _v2OffsetPositionOpen - new Vector2(4, 0), ref _iAnimCurrentFrame, ref _iAnimCurrentTime, Microsoft.Xna.Framework.Graphics.SpriteEffects.FlipHorizontally);
                    else
                        _bAnimationOver = Visu.Instance.displayAnimation(_iIdAnimOpening, ScreenPosition, ref _iAnimCurrentFrame, ref _iAnimCurrentTime);
                    break;
            }
        }

        public override void die()
        {
            _shape._bCollisionEnable = false;
            _bActive = false;
            Manager.ExplosionManager.Instance.activate(_iIdAnimationExplosion, Position + _v2PositionExplosionOffset);
        }

        public override void hurt(int damages)
        {
            if (_innerState != eState.eStateOpen) return;

            HP -= damages;
            if(HP <= 0) die();
        }
    }
}
